#!/bin/sh

set -eu

csv2uos() {
	awk -F, -vOFMT='%0.17f' '{print -$1, -$3, $2}'
}
csv2pose() {
	awk -F, -vOFMT='%0.17f' 'BEGIN{pi=atan2(0,-1)} !/^$/ {print -$1, -$3, $2; print ($4*180)/pi, ($6*180)/pi, (-$5*180)/pi}'
}
pose2csv() {
	awk -vOFS=',' -vFS="[\n ]+" -vRS="" -vOFMT='%0.10f' 'BEGIN{pi=atan2(0,-1)} {print -$1, $3, -$2, $4*pi/180, -$6*pi/180, $5*pi/180}'
}
calcstats() {
	awk 'BEGIN{fn=0;tp=0;tn=0;fp=0} { if ($1 == 1 && $2 == 1) {tp+=1} else if ($1 == 0 && $2 == 0) {tn+=1} else if ($1 == 1 && $2 == 0) {fn+=1} else if ($1 == 0 && $2 == 1) {fp+=1}} END{p=(tp+fp==0?0:tp/(tp+fp));r=(tp+fn==0?0:tp/(tp+fn));print tp, tn, fp, fn, p, r, (p+r==0?0:2*p*r/(p+r))}'
}
collectstats() {
	awk 'BEGIN{tp=tn=fp=fn=0} {tp+=$1;tn+=$2;fp+=$3;fn+=$4} END{p=tp/(tp+fp);r=tp/(tp+fn);printf "%.02f ", (2*p*r/(p+r))}'
}

download() {
	if [ ! -e 3dtk ]; then
		svn checkout -r 1535 https://svn.code.sf.net/p/slam6d/code/trunk 3dtk
	fi

	if [ ! -e comma ]; then
		git clone https://github.com/acfr/comma.git comma
		git -C comma reset --hard 7a1dba361524543e33b03d5987bde377b4c077bb
	fi

	if [ ! -e snark ]; then
		git clone https://github.com/acfr/snark.git snark
		git -C snark reset --hard be6543324942a659ab64184434b0698488522ca6
	fi

	if [ ! -e data/underwood ]; then
		wget -nc http://perception.acfr.usyd.edu.au/demos/change_detection/change-detection-package.zip
		echo 5d86adff374bc9662f40080da9d46f2d4ea1323ff9c34d133a698fffc5c5f1dc  change-detection-package.zip | sha256sum --check -
		mkdir -p data/underwood
		unzip -d data/underwood change-detection-package.zip
		rm change-detection-package.zip
	fi

	if [ ! -e data/wue_city ]; then
		wget -nc http://kos.informatik.uni-osnabrueck.de/3Dscans/wue_city.tar.xz
		echo fb078293ba0acca2759fdb668a80d27d3a5914e4bc1987bd426fa8c8a6950767  wue_city.tar.xz | sha256sum --check -
		mkdir -p data/wue_city
		tar --strip-components=1 -C data/wue_city -xf wue_city.tar.xz
		rm wue_city.tar.xz
	fi

	if [ ! -e data/lecturehall ]; then
		wget -nc http://kos.informatik.uni-osnabrueck.de/3Dscans/lecturehall.tar.xz
		echo 3095ad4f7e6d77828b410430c46d684daf187d70ff8b74861352db897d541c35  lecturehall.tar.xz | sha256sum --check -
		mkdir -p data/lecturehall
		tar --strip-components=1 -C data/lecturehall -xf lecturehall.tar.xz
		rm lecturehall.tar.xz
	fi
}

compile() {
	mkdir -p 3dtk.build
	cmake -B3dtk.build -H3dtk -DWITH_PEOPLEREMOVER=ON -DWITH_FBR=ON -DWITH_TOOLS=ON
	make -C 3dtk.build

	mkdir -p comma.build comma.install
	cmake -B"$PWD/comma.build" -H"$PWD/comma" -DCMAKE_INSTALL_PREFIX="$PWD/comma.install" -DINSTALL_BASH_COMPLETION=OFF -DPYTHON_PACKAGE_INSTALL_PREFIX="$PWD/comma.install"
	make -C comma.build
	make -C comma.build install

	mkdir -p snark.build snark.install
	cmake -B"$PWD/snark.build" -H"$PWD/snark" -DCMAKE_INSTALL_PREFIX="$PWD/snark.install" -DINSTALL_BASH_COMPLETION=OFF -Dcomma_DIR="$PWD/comma.install/CMakeFiles" -Dsnark_build_graphics=OFF
	make -C snark.build
	make -C snark.build install
}

convert() {
	mkdir -p data/underwood/sim/3d
	csv2uos < data/underwood/sim/sim1.pose1.object1.label.csv > data/underwood/sim/3d/scan000.3d
	csv2uos < data/underwood/sim/sim1.pose1.object2.label.csv > data/underwood/sim/3d/scan001.3d
	csv2uos < data/underwood/sim/sim1.pose2.object1.label.csv > data/underwood/sim/3d/scan002.3d
	csv2uos < data/underwood/sim/sim1.pose2.object2.label.csv > data/underwood/sim/3d/scan003.3d
	csv2uos < data/underwood/sim/sim1.pose3.object1.label.csv > data/underwood/sim/3d/scan004.3d
	csv2uos < data/underwood/sim/sim1.pose3.object2.label.csv > data/underwood/sim/3d/scan005.3d
	csv2uos < data/underwood/sim/sim1.pose4.object1.label.csv > data/underwood/sim/3d/scan006.3d
	csv2uos < data/underwood/sim/sim1.pose4.object2.label.csv > data/underwood/sim/3d/scan007.3d
	csv2pose < data/underwood/sim/sim1.pose1.csv > data/underwood/sim/3d/scan000.pose
	csv2pose < data/underwood/sim/sim1.pose1.csv > data/underwood/sim/3d/scan001.pose
	csv2pose < data/underwood/sim/sim1.pose2.csv > data/underwood/sim/3d/scan002.pose
	csv2pose < data/underwood/sim/sim1.pose2.csv > data/underwood/sim/3d/scan003.pose
	csv2pose < data/underwood/sim/sim1.pose3.csv > data/underwood/sim/3d/scan004.pose
	csv2pose < data/underwood/sim/sim1.pose3.csv > data/underwood/sim/3d/scan005.pose
	csv2pose < data/underwood/sim/sim1.pose4.csv > data/underwood/sim/3d/scan006.pose
	csv2pose < data/underwood/sim/sim1.pose4.csv > data/underwood/sim/3d/scan007.pose

	mkdir -p data/underwood/lab/3d
	csv2uos < data/underwood/lab/lab1.pose1.object1.label.csv > data/underwood/lab/3d/scan000.3d
	csv2uos < data/underwood/lab/lab1.pose1.object2.label.csv > data/underwood/lab/3d/scan001.3d
	csv2uos < data/underwood/lab/lab1.pose1.object3.label.csv > data/underwood/lab/3d/scan002.3d
	csv2uos < data/underwood/lab/lab1.pose1.object4.label.csv > data/underwood/lab/3d/scan003.3d
	csv2uos < data/underwood/lab/lab1.pose2.object1.label.csv > data/underwood/lab/3d/scan004.3d
	csv2uos < data/underwood/lab/lab1.pose2.object2.label.csv > data/underwood/lab/3d/scan005.3d
	csv2uos < data/underwood/lab/lab1.pose2.object3.label.csv > data/underwood/lab/3d/scan006.3d
	csv2uos < data/underwood/lab/lab1.pose2.object4.label.csv > data/underwood/lab/3d/scan007.3d
	csv2uos < data/underwood/lab/lab1.pose3.object1.label.csv > data/underwood/lab/3d/scan008.3d
	csv2uos < data/underwood/lab/lab1.pose3.object2.label.csv > data/underwood/lab/3d/scan009.3d
	csv2uos < data/underwood/lab/lab1.pose3.object3.label.csv > data/underwood/lab/3d/scan010.3d
	csv2uos < data/underwood/lab/lab1.pose3.object4.label.csv > data/underwood/lab/3d/scan011.3d
	csv2pose < data/underwood/lab/lab1.pose1.csv > data/underwood/lab/3d/scan000.pose
	csv2pose < data/underwood/lab/lab1.pose1.csv > data/underwood/lab/3d/scan001.pose
	csv2pose < data/underwood/lab/lab1.pose1.csv > data/underwood/lab/3d/scan002.pose
	csv2pose < data/underwood/lab/lab1.pose1.csv > data/underwood/lab/3d/scan003.pose
	csv2pose < data/underwood/lab/lab1.pose2.csv > data/underwood/lab/3d/scan004.pose
	csv2pose < data/underwood/lab/lab1.pose2.csv > data/underwood/lab/3d/scan005.pose
	csv2pose < data/underwood/lab/lab1.pose2.csv > data/underwood/lab/3d/scan006.pose
	csv2pose < data/underwood/lab/lab1.pose2.csv > data/underwood/lab/3d/scan007.pose
	csv2pose < data/underwood/lab/lab1.pose3.csv > data/underwood/lab/3d/scan008.pose
	csv2pose < data/underwood/lab/lab1.pose3.csv > data/underwood/lab/3d/scan009.pose
	csv2pose < data/underwood/lab/lab1.pose3.csv > data/underwood/lab/3d/scan010.pose
	csv2pose < data/underwood/lab/lab1.pose3.csv > data/underwood/lab/3d/scan011.pose

	mkdir -p data/lecturehall/3d
	csv2uos < data/lecturehall/lecturehall1.pose1.object1.label.csv > data/lecturehall/3d/scan000.3d
	csv2pose < data/lecturehall/lecturehall1.pose1.csv > data/lecturehall/3d/scan000.pose
	csv2uos < data/lecturehall/lecturehall1.pose2.object2.label.csv > data/lecturehall/3d/scan001.3d
	csv2pose < data/lecturehall/lecturehall1.pose2.csv > data/lecturehall/3d/scan001.pose

	mkdir -p data/underwood/carpark/3d
	csv2uos < data/underwood/carpark/carpark1.pose1.object1.label.csv > data/underwood/carpark/3d/scan000.3d
	csv2uos < data/underwood/carpark/carpark1.pose1.object2.label.csv > data/underwood/carpark/3d/scan001.3d
	csv2uos < data/underwood/carpark/carpark1.pose2.object2.label.csv > data/underwood/carpark/3d/scan002.3d
	csv2uos < data/underwood/carpark/carpark1.pose3.object2.label.csv > data/underwood/carpark/3d/scan003.3d
	csv2pose < data/underwood/carpark/carpark1.pose1.csv > data/underwood/carpark/3d/scan000.pose
	csv2pose < data/underwood/carpark/carpark1.pose1.csv > data/underwood/carpark/3d/scan001.pose
	csv2pose < data/underwood/carpark/carpark1.pose2.csv > data/underwood/carpark/3d/scan002.pose
	csv2pose < data/underwood/carpark/carpark1.pose3.csv > data/underwood/carpark/3d/scan003.pose

	mkdir -p data/underwood/carpark.registered/3d
	ln -sf ../../carpark/3d/scan000.3d data/underwood/carpark.registered/3d/scan000.3d
	ln -sf ../../carpark/3d/scan001.3d data/underwood/carpark.registered/3d/scan001.3d
	ln -sf ../../carpark/3d/scan002.3d data/underwood/carpark.registered/3d/scan002.3d
	ln -sf ../../carpark/3d/scan003.3d data/underwood/carpark.registered/3d/scan003.3d
	cp data/underwood/carpark/3d/scan000.pose data/underwood/carpark.registered/3d/scan000.pose
	cp data/underwood/carpark/3d/scan001.pose data/underwood/carpark.registered/3d/scan001.pose
	cp data/underwood/carpark/3d/scan002.pose data/underwood/carpark.registered/3d/scan002.pose
	cp data/underwood/carpark/3d/scan003.pose data/underwood/carpark.registered/3d/scan003.pose
	3dtk/bin/slam6D --cldist=5 --dist=0.25 --epsSLAM=0.005 --loopclosefile=/dev/null data/underwood/carpark.registered/3d
	3dtk/bin/frames2pose -s 0 -e 3 data/underwood/carpark.registered/3d
	rm data/underwood/carpark.registered/3d/scan000.frames
	rm data/underwood/carpark.registered/3d/scan001.frames
	rm data/underwood/carpark.registered/3d/scan002.frames
	rm data/underwood/carpark.registered/3d/scan003.frames

	ln -sf ../carpark/carpark1.pose1.object1.label.csv data/underwood/carpark.registered/carpark.registered1.pose1.object1.label.csv
	ln -sf ../carpark/carpark1.pose1.object2.label.csv data/underwood/carpark.registered/carpark.registered1.pose1.object2.label.csv
	ln -sf ../carpark/carpark1.pose2.object2.label.csv data/underwood/carpark.registered/carpark.registered1.pose2.object2.label.csv
	ln -sf ../carpark/carpark1.pose3.object2.label.csv data/underwood/carpark.registered/carpark.registered1.pose3.object2.label.csv
	pose2csv < data/underwood/carpark.registered/3d/scan000.pose > data/underwood/carpark.registered/carpark.registered1.pose1.csv
	pose2csv < data/underwood/carpark.registered/3d/scan002.pose > data/underwood/carpark.registered/carpark.registered1.pose2.csv
	pose2csv < data/underwood/carpark.registered/3d/scan003.pose > data/underwood/carpark.registered/carpark.registered1.pose3.csv
}

clean() {
	rm data/lecturehall/3d/scan000.3d \
		data/lecturehall/3d/scan000.pose \
		data/lecturehall/3d/scan001.3d \
		data/lecturehall/3d/scan001.pose \
		data/lecturehall/lecturehall1.pose1.csv \
		data/lecturehall/lecturehall1.pose1.object1.label.csv \
		data/lecturehall/lecturehall1.pose2.csv \
		data/lecturehall/lecturehall1.pose2.object2.label.csv \
		data/underwood/carpark/3d/scan000.3d \
		data/underwood/carpark/3d/scan000.pose \
		data/underwood/carpark/3d/scan001.3d \
		data/underwood/carpark/3d/scan001.pose \
		data/underwood/carpark/3d/scan002.3d \
		data/underwood/carpark/3d/scan002.pose \
		data/underwood/carpark/3d/scan003.3d \
		data/underwood/carpark/3d/scan003.pose \
		data/underwood/carpark/carpark1.pose1.csv \
		data/underwood/carpark/carpark1.pose1.object1.label.csv \
		data/underwood/carpark/carpark1.pose1.object2.label.csv \
		data/underwood/carpark/carpark1.pose2.csv \
		data/underwood/carpark/carpark1.pose2.object2.label.csv \
		data/underwood/carpark/carpark1.pose3.csv \
		data/underwood/carpark/carpark1.pose3.object2.label.csv \
		data/underwood/carpark.registered/3d/scan000.pose \
		data/underwood/carpark.registered/3d/scan001.pose \
		data/underwood/carpark.registered/3d/scan002.pose \
		data/underwood/carpark.registered/3d/scan003.pose \
		data/underwood/carpark.registered/3d/scan000.3d \
		data/underwood/carpark.registered/3d/scan001.3d \
		data/underwood/carpark.registered/3d/scan002.3d \
		data/underwood/carpark.registered/3d/scan003.3d \
		data/underwood/carpark.registered/carpark.registered1.pose1.csv \
		data/underwood/carpark.registered/carpark.registered1.pose2.csv \
		data/underwood/carpark.registered/carpark.registered1.pose3.csv \
		data/underwood/carpark.registered/carpark.registered1.pose1.object1.label.csv \
		data/underwood/carpark.registered/carpark.registered1.pose1.object2.label.csv \
		data/underwood/carpark.registered/carpark.registered1.pose2.object2.label.csv \
		data/underwood/carpark.registered/carpark.registered1.pose3.object2.label.csv \
		data/underwood/detect-change-cluster.sh \
		data/underwood/detect-change.sh \
		data/underwood/detect-change-streaming.sh \
		data/underwood/lab/3d/scan000.3d \
		data/underwood/lab/3d/scan000.pose \
		data/underwood/lab/3d/scan001.3d \
		data/underwood/lab/3d/scan001.pose \
		data/underwood/lab/3d/scan002.3d \
		data/underwood/lab/3d/scan002.pose \
		data/underwood/lab/3d/scan003.3d \
		data/underwood/lab/3d/scan003.pose \
		data/underwood/lab/3d/scan004.3d \
		data/underwood/lab/3d/scan004.pose \
		data/underwood/lab/3d/scan005.3d \
		data/underwood/lab/3d/scan005.pose \
		data/underwood/lab/3d/scan006.3d \
		data/underwood/lab/3d/scan006.pose \
		data/underwood/lab/3d/scan007.3d \
		data/underwood/lab/3d/scan007.pose \
		data/underwood/lab/3d/scan008.3d \
		data/underwood/lab/3d/scan008.pose \
		data/underwood/lab/3d/scan009.3d \
		data/underwood/lab/3d/scan009.pose \
		data/underwood/lab/3d/scan010.3d \
		data/underwood/lab/3d/scan010.pose \
		data/underwood/lab/3d/scan011.3d \
		data/underwood/lab/3d/scan011.pose \
		data/underwood/lab/lab1.pose1.csv \
		data/underwood/lab/lab1.pose1.object1.label.csv \
		data/underwood/lab/lab1.pose1.object2.label.csv \
		data/underwood/lab/lab1.pose1.object3.label.csv \
		data/underwood/lab/lab1.pose1.object4.label.csv \
		data/underwood/lab/lab1.pose2.csv \
		data/underwood/lab/lab1.pose2.object1.label.csv \
		data/underwood/lab/lab1.pose2.object2.label.csv \
		data/underwood/lab/lab1.pose2.object3.label.csv \
		data/underwood/lab/lab1.pose2.object4.label.csv \
		data/underwood/lab/lab1.pose3.csv \
		data/underwood/lab/lab1.pose3.object1.label.csv \
		data/underwood/lab/lab1.pose3.object2.label.csv \
		data/underwood/lab/lab1.pose3.object3.label.csv \
		data/underwood/lab/lab1.pose3.object4.label.csv \
		data/underwood/lab-motion/change-detection-raw-velodyne.bin \
		data/underwood/lab-motion/db.firmware_404.xml \
		data/underwood/readme.txt \
		data/underwood/sim/3d/scan000.3d \
		data/underwood/sim/3d/scan000.pose \
		data/underwood/sim/3d/scan001.3d \
		data/underwood/sim/3d/scan001.pose \
		data/underwood/sim/3d/scan002.3d \
		data/underwood/sim/3d/scan002.pose \
		data/underwood/sim/3d/scan003.3d \
		data/underwood/sim/3d/scan003.pose \
		data/underwood/sim/3d/scan004.3d \
		data/underwood/sim/3d/scan004.pose \
		data/underwood/sim/3d/scan005.3d \
		data/underwood/sim/3d/scan005.pose \
		data/underwood/sim/3d/scan006.3d \
		data/underwood/sim/3d/scan006.pose \
		data/underwood/sim/3d/scan007.3d \
		data/underwood/sim/3d/scan007.pose \
		data/underwood/sim/sim1.pose1.csv \
		data/underwood/sim/sim1.pose1.object1.label.csv \
		data/underwood/sim/sim1.pose1.object2.label.csv \
		data/underwood/sim/sim1.pose2.csv \
		data/underwood/sim/sim1.pose2.object1.label.csv \
		data/underwood/sim/sim1.pose2.object2.label.csv \
		data/underwood/sim/sim1.pose3.csv \
		data/underwood/sim/sim1.pose3.object1.label.csv \
		data/underwood/sim/sim1.pose3.object2.label.csv \
		data/underwood/sim/sim1.pose4.csv \
		data/underwood/sim/sim1.pose4.object1.label.csv \
		data/underwood/sim/sim1.pose4.object2.label.csv \
		data/underwood/Underwood-ICRA13-ChangeDetection.pdf \
		data/wue_city/scan000.3d \
		data/wue_city/scan000.pose \
		data/wue_city/scan001.3d \
		data/wue_city/scan001.pose \
		data/wue_city/scan002.3d \
		data/wue_city/scan002.pose \
		data/wue_city/scan003.3d \
		data/wue_city/scan003.pose \
		data/wue_city/scan004.3d \
		data/wue_city/scan004.pose \
		data/wue_city/scan005.3d \
		data/wue_city/scan005.pose
	rm -d data/lecturehall/3d \
		data/lecturehall \
		data/underwood/carpark/3d \
		data/underwood/carpark \
		data/underwood/carpark.registered/3d \
		data/underwood/carpark.registered \
		data/underwood/lab/3d \
		data/underwood/lab \
		data/underwood/lab-motion \
		data/underwood/sim/3d \
		data/underwood/sim \
		data/underwood \
		data/wue_city \
		data

	echo "please clean the following directories manually:"
	echo
	echo "snark/ snark.install/ snark.build/ comma/ comma.install/ comma.build/ 3dtk/ 3dtk.build/"
}

export PATH="$PWD/comma.install/bin/":"$PWD/snark.install/bin":"$PATH"
export PYTHONPATH="$PWD/comma.install/lib/python2.7/site-packages/"

eval_underwood() {
	path="$1"
	n="${path##underwood/}"
	ANGLE_THRESHOLD="$(math-deg2rad $2 | tr -d '\r\n')"
	RANGE_THRESHOLD="$3"
	printf "%0.1f %5.2f " $2 $3
	scans=
	for f in "data/$path/${n}1"*pose?.object?.label.csv; do
		bn="${f##*/${n}1.}"
		bn="${bn%.label.csv}"
		scans="$scans $bn"
	done
	for scan1 in $scans; do
		csv-to-bin 3d,2ui < data/$path/${n}1.$scan1.label.csv > data/$path/${n}1.$scan1.label.csv.bin
	done
	for scan1 in $scans; do
		for scan2 in $scans; do
			if [ ${scan1#pose?} = ${scan2#pose?} ]; then continue; fi
			INPUT_FILE_1="data/$path/${n}1.$scan1.label.csv.bin"
			INPUT_FILE_2="data/$path/${n}1.$scan2.label.csv.bin"
			POSE1="$(tr -d "\r\n" < data/$path/${n}1.${scan1%.object*}.csv)"
			POSE2="$(tr -d "\r\n" < data/$path/${n}1.${scan2%.object*}.csv)"
			cat $INPUT_FILE_2 | points-to-polar --binary=3d,2ui --fields=x,y,z > ref2rbe.bin
			cat $INPUT_FILE_1 \
			  | points-frame --binary=3d,2ui --fields=x,y,z --from=$POSE1 --to=$POSE2 \
			  | points-to-polar --binary=3d,2ui --fields=x,y,z \
			  | points-detect-change --binary=3d,2ui --fields=r,b,e --angle-threshold=$ANGLE_THRESHOLD --range-threshold=$RANGE_THRESHOLD ref2rbe.bin \
			  | points-to-cartesian --binary=3d,2ui --fields=r,b,e \
			  | points-frame --binary=3d,2ui --fields=x,y,z --from=$POSE2 \
			  > additions.bin
			cat $INPUT_FILE_1 | points-frame --binary=3d,2ui --fields=x,y,z --from=$POSE1 > georef1.bin
			addition_object_points_count=$( cat georef1.bin | csv-eval --binary 3d,2ui --fields x,y,z,scan,label --output-if "label==1" | csv-from-bin 3d,2ui | wc -l )
			addition_non_object_points_count=$( cat georef1.bin | csv-eval --binary 3d,2ui --fields x,y,z,scan,label --output-if "label==0" | csv-from-bin 3d,2ui | wc -l )
			addition_true_positives=$( cat additions.bin | csv-eval --binary 3d,2ui,3d,2ui --fields x,y,z,scan,label --output-if "label==1" | csv-from-bin 3d,2ui,3d,2ui | wc -l )
			addition_false_positives=$( cat additions.bin | csv-eval --binary 3d,2ui,3d,2ui --fields x,y,z,scan,label --output-if "label==0" | csv-from-bin 3d,2ui,3d,2ui | wc -l )
			echo "$addition_true_positives $addition_non_object_points_count $addition_false_positives $addition_object_points_count" \
			| awk '{tp=$1; fp=$3; tn=$2-fp; fn=$4-tp; p=(tp+fp==0?0:tp/(tp+fp));r=(tp+fn==0?0:tp/(tp+fn)); print tp, tn, fp, fn, p, r, (p+r==0?0:2*p*r/(p+r))}'
			rm additions.bin georef1.bin ref2rbe.bin;
		done
	done | collectstats
	for scan1 in $scans; do
		rm "data/$path/${n}1.$scan1.label.csv.bin"
	done
}

eval_3dtk() {
	path=$1
	n="${path##underwood/}"
	start=$2
	end=$3
	voxel=$4
	normals=$(3dtk/bin/peopleremover \
		-s $start -e $end -f uos \
		--maskdir=data/$path/3d/pplremover \
		--min-cluster-size=0 \
		--no-subvoxel-accuracy \
		--voxel-size=$voxel \
		--maxrange-method=normals \
		--normal-method=angle \
		data/$path/3d 2>&1 \
		| sed -ne 's/Computed normals for [0-9]\+ points (\([0-9.]\+\)% of all points in scan)/\1/p' \
		| awk '{sum+=$1} END{print sum/('$end'-'$start'+1)}')
	printf "%6.3f %0.2f " $voxel $normals
	scans=
	for f in "data/$path/${n}1"*pose?.object?.label.csv; do
		bn="${f##*/${n}1.}"
		bn="${bn%.label.csv}"
		scans="$scans $bn"
	done
	i=0
	for scan in $scans; do
		mask=$(printf "data/$path/3d/pplremover/scan%03d.mask" $i)
		cut -d, -f 5 data/$path/${n}1.$scan.label.csv | tr -d '\r' | paste - "$mask" | calcstats
		rm "$mask"
		i=$((i+1))
	done | collectstats
	rm -d "data/$path/3d/pplremover/"
	rm scan000.3d scan000.pose scan001.3d scan001.pose
}

eval_all() {
	printf "dataset     | %-21s | %s\n" "underwood" "3dtk"
	printf "            | Ta  Tr    f1   t(s)   | VS     N    f1   t(s)\n"
	printf -- "----------- | --------------------- | ------------------------\n"
	r1=$(/usr/bin/time --format "%e" $script eval_underwood underwood/sim 1.4 0.1 2>&1)
	r2=$(/usr/bin/time --format "%e" $script eval_3dtk underwood/sim 0 7 0.6 2>&1)
	printf "sim         | %-21s | %s\n" "$r1" "$r2"
	r1=$(/usr/bin/time --format "%e" $script eval_underwood underwood/lab 1.2 0.2 2>&1)
	r2=$(/usr/bin/time --format "%e" $script eval_3dtk underwood/lab 0 11 0.1 2>&1)
	printf "lab         | %-21s | %s\n" "$r1" "$r2"
	r1=$(/usr/bin/time --format "%e" $script eval_underwood underwood/carpark.registered 1.0 0.35 2>&1)
	r2=$(/usr/bin/time --format "%e" $script eval_3dtk underwood/carpark.registered 0 3 0.125 2>&1)
	printf "carpark     | %-21s | %s\n" "$r1" "$r2"
	r1=$(/usr/bin/time --format "%e" $script eval_underwood lecturehall 0.8 30 2>&1)
	r2=$(/usr/bin/time --format "%e" $script eval_3dtk lecturehall 0 1 10 2>&1)
	printf "lecturehall | %-21s | %s\n" "$r1" "$r2"
	printf -- "----------- | --------------------- | ------------------------\n"
	echo
	echo "     Ta:   angle threshold"
	echo "     Tr:   range threshold"
	echo "     f1:   f1-score"
	echo "     t(s): runtime in seconds"
	echo "     VS:   voxel size"
	echo "     N:    percent of points for which the normal vector got calculated"
	echo
}

if [ $# -lt 1 ]; then
	echo "usage: $0 action"
	echo
	echo "available actions:"
	echo
	echo "   download - downloads all data and code"
	echo "   compile  - compiles all code"
	echo "   convert  - converts all data"
	echo "   eval     - run change detection"
	echo "   all      - all of the above"
	exit 1
fi

script=$0
mode=$1
shift
case $mode in
	eval_3dtk)
		eval_3dtk $@;;
	eval_underwood)
		eval_underwood $@;;
	convert)
		convert;;
	download)
		download;;
	compile)
		compile;;
	clean)
		clean;;
	eval)
		eval_all;;
	all)
		download
		compile
		convert
		eval_all
		;;
	*)
		echo "invalid mode: $mode" >&2;;
esac
