#!/bin/sh

set -eu

export PATH="$PWD/comma.install/bin/":"$PWD/snark.install/bin":"$PWD/3dtk/bin":"$PATH"
export PYTHONPATH="$PWD/comma.install/lib/python2.7/site-packages/":"$PWD/3dtk/lib"

urlencode()
{
	printf "$1" | curl -Gso /dev/null -w %{url_effective} --data-urlencode @- "" | cut -c 3-
}

pose2csv() {
	awk -vOFS=',' -vFS="[\n ]+" -vRS="" -vOFMT='%0.10f' 'BEGIN{pi=atan2(0,-1)} {print -$1, $3, -$2, $4*pi/180, -$6*pi/180, $5*pi/180}'
}
collectstats() {
	awk 'BEGIN{tp=tn=fp=fn=0} {tp+=$1;tn+=$2;fp+=$3;fn+=$4} END{p=(tp+fp==0?0:tp/(tp+fp));r=(tp+fn==0?0:tp/(tp+fn));printf "%.04f ", (p+r==0?0:2*p*r/(p+r))}'
}
calcstats() {
	awk 'BEGIN{fn=0;tp=0;tn=0;fp=0} { if ($1 == 1 && $2 == 1) {tp+=1} else if ($1 == 0 && $2 == 0) {tn+=1} else if ($1 == 1 && $2 == 0) {fn+=1} else if ($1 == 0 && $2 == 1) {fp+=1}} END{p=(tp+fp==0?0:tp/(tp+fp));r=(tp+fn==0?0:tp/(tp+fn));print tp, tn, fp, fn, p, r, (p+r==0?0:2*p*r/(p+r))}'
}

day=2011_09_26
ids="1 2 5 9 11 13 14 15 17 18 19 20 22 23 27 28 29 32 35 36 39 46 48 51 52 56 57 59 60 61 64 70 79 84 86 87 91 93"

md5sums="
9ef3045154ddeb0f80c1983bc6e10149  2011_09_26_drive_0001_sync.zip
d1b37eef02ad3369ad872705e2dcbdc2  2011_09_26_drive_0002_sync.zip
320fd5f05862be735c9bfecff70ba6cc  2011_09_26_drive_0005_sync.zip
337be8431b438ece5c64539ba3ec41f8  2011_09_26_drive_0009_sync.zip
aed4a680947065d6c3ee698f207e4e6c  2011_09_26_drive_0011_sync.zip
f41767ce0a066f20096b7668e2b5c0a1  2011_09_26_drive_0013_sync.zip
a422e0b058f46703d1cceb9bce49e41f  2011_09_26_drive_0014_sync.zip
e08b2b7f7872c15c5a4be051e8c10765  2011_09_26_drive_0015_sync.zip
ed7ea3885148c39dd1ab2a690cdf983b  2011_09_26_drive_0017_sync.zip
2b3c659c983d562c2b09ec1cfa322c8b  2011_09_26_drive_0018_sync.zip
0c42c61596f6dcbda9319568e9d4c0e9  2011_09_26_drive_0019_sync.zip
f77f55a248c420dd7296e3489aa9ef8a  2011_09_26_drive_0020_sync.zip
a17b3bf589a6fda6a0088ba1ff65a680  2011_09_26_drive_0022_sync.zip
8980233212a6cb0dceea042b5e1807c1  2011_09_26_drive_0023_sync.zip
c9cccf42e271dcfb9c82b2c1f1f2e82a  2011_09_26_drive_0027_sync.zip
2f13f329de12c6f36acd9272d9e010a0  2011_09_26_drive_0028_sync.zip
83946fb9921cf7c58664df7709ab166c  2011_09_26_drive_0029_sync.zip
c30eea833603a53e7b42e79f773c0901  2011_09_26_drive_0032_sync.zip
d47d613f66f40e471d243dcc46c0e94a  2011_09_26_drive_0035_sync.zip
67c839098cf6366838955960d7e7fcf3  2011_09_26_drive_0036_sync.zip
06d46c2d77feab5219707e67dcd62014  2011_09_26_drive_0039_sync.zip
dab20aa2ff34ad6ae0a50987378a3af5  2011_09_26_drive_0046_sync.zip
6a7c75b43cc48b607ee11446d90a4d9d  2011_09_26_drive_0048_sync.zip
977fcc103557744a2f69d38b7bb7a012  2011_09_26_drive_0051_sync.zip
bc79a8aa191ab05d4f56c916b9df5336  2011_09_26_drive_0052_sync.zip
0f93c65f9473fe6095fcc26ae5c17e4d  2011_09_26_drive_0056_sync.zip
10eed940384c8470c3b86fab640af9c7  2011_09_26_drive_0057_sync.zip
03827bbe1fee826ada8c889b42544707  2011_09_26_drive_0059_sync.zip
9039dd91100c8c822b2d2212af1e3df2  2011_09_26_drive_0060_sync.zip
354c6fb62d72ea305f721850bf400567  2011_09_26_drive_0061_sync.zip
34cd4139ea33f29e8be0c1dc7c2412d9  2011_09_26_drive_0064_sync.zip
ccad2e2bd7a58f65a81beee2632f73ab  2011_09_26_drive_0070_sync.zip
80f040f9919b8fbb049da82420577897  2011_09_26_drive_0079_sync.zip
04e25ef6741f2dea19b57d83abc2f012  2011_09_26_drive_0084_sync.zip
d0d3dcd1816ef4c5a9323cdb47430eed  2011_09_26_drive_0086_sync.zip
f34f5a2312b0a38a27f42859a504ab95  2011_09_26_drive_0087_sync.zip
97e0c9fd9a14cd541afd0f5b466ebe50  2011_09_26_drive_0091_sync.zip
2c13ef06b643eb9254748b92baad84a6  2011_09_26_drive_0093_sync.zip
"

download()
{
	if [ ! -e Extended_MOD_Masks ]; then
		fname=Extended_KittiMoSeg_MOD_Masks.zip
		if [ ! -e "$fname" ]; then
			GDRIVE_FILE_ID=1kuLehp9gB5xkdxcxSOHXqRFhas4f8Uke
			curl -c ./cookie -s -L "https://drive.google.com/uc?export=download&id=${GDRIVE_FILE_ID}" > /dev/null
			CONFIRM_ID=$(awk '/download/ {print $NF}' ./cookie)
			curl -Lb ./cookie "https://drive.google.com/uc?export=download&confirm=${CONFIRM_ID}&id=${GDRIVE_FILE_ID}" -o "$fname"
			rm ./cookie
		fi
		echo "868cb7169b9d5295728cb54cfd1a752e  $fname" | md5sum --check
		unzip "$fname"
	fi

	if [ ! -e "$day/calib_cam_to_cam.txt" ]; then
		fname="${day}_calib.zip"
		if [ ! -e "$fname" ]; then
			wget "https://s3.eu-central-1.amazonaws.com/avg-kitti/raw_data/$fname"
		fi
		echo "5d6cfb6bd7ec7c5f5231d16ead7ce9f1  $fname" | md5sum --check
		unzip "$fname"
	fi

	for i in $ids; do
		fname="${day}_drive_$(printf "%04d" $i)_sync.zip"
		if [ -e "$fname" ]; then
			continue
		fi
		dir="${day}_drive_$(printf "%04d" $i)"
		wget "https://s3.eu-central-1.amazonaws.com/avg-kitti/raw_data/${dir}/${fname}"
		echo "$md5sums" | grep "$fname" | md5sum --check
		path="${day}/${day}_drive_$(printf "%04d" $i)_sync/"
		if [ ! -e "$path" ]; then
			fname="${day}_drive_$(printf "%04d" $i)_sync.zip"
			unzip "$fname"
		fi
	done

	if [ ! -e 3dtk ]; then
		svn checkout -r 2248 https://svn.code.sf.net/p/slam6d/code/trunk 3dtk
	fi

	if [ ! -e comma ]; then
		#git clone https://github.com/acfr/comma.git comma
		#git clone https://github.com/3DTK/comma.git comma
		#git clone https://gitlab.com/orthographic/comma.git comma
		git clone https://gitlab.com/patteone/comma.git comma
		git -C comma reset --hard e614d48a9caf03057581ea2bc2256b8adafb1cc6
	fi

	if [ ! -e snark ]; then
		#git clone https://github.com/acfr/snark.git snark
		#git clone https://github.com/3DTK/snark.git snark
		#git clone https://gitlab.com/orthographic/snark.git snark
		git clone https://gitlab.com/patteone/snark.git snark
		git -C snark reset --hard 580c4a176bae62a75e0002df4f954fc433326aa8
	fi
}

compile() {
	mkdir -p 3dtk.build
	cmake -B3dtk.build -H3dtk -DWITH_PYTHON=ON
	make -C 3dtk.build

	mkdir -p comma.build comma.install
	cmake -B"$PWD/comma.build" -H"$PWD/comma" -DCMAKE_INSTALL_PREFIX="$PWD/comma.install" -DINSTALL_BASH_COMPLETION=OFF -DPYTHON_PACKAGE_INSTALL_PREFIX="$PWD/comma.install"
	make -C comma.build
	make -C comma.build install

	mkdir -p snark.build snark.install
	cmake -B"$PWD/snark.build" -H"$PWD/snark" -DCMAKE_INSTALL_PREFIX="$PWD/snark.install" -DINSTALL_BASH_COMPLETION=OFF -Dcomma_DIR="$PWD/comma.install/CMakeFiles" -Dsnark_build_graphics=OFF
	make -C snark.build
	make -C snark.build install
}

convert() {
	for i in $ids; do
		#./3dtk.build/src/peopleremover/kitti2scan.py $day $i
		./kitti2scan.py $day $i
		dirname="$day/${day}_drive_$(printf "%04d" $i)_sync"
		#slam6D -m 5000 -d 50.0 --metascan -f uosr --epsICP=0.000001 -a 2 "$dirname/3dtk"
		slam6D -f uosr -r 10 --octree=1 -G 1 --epsICP=0.000000001 --epsSLAM=0.000000001 -a 2 -d 25 -i 50 -I 50 -n "$dirname/3dtk/slam6d.net" "$dirname/3dtk"
		frames2pose "$dirname/3dtk"
		cp "$dirname/3dtk/"*.pose "$dirname/changedetection/"
		for scan in "$dirname/changedetection"/*.3d; do
			binname=$(basename "$scan" .3d).bin
			echo "convert to binary: $scan" >&2
			while read x y z r; do
				# to transform hexfloat to decimal float
				printf "%.017f %.017f %.017f %.017f\n" $x $y $z $r
			done < "$scan" \
				| awk -vOFS=',' -vOFMT='%.17f' '{ print -$1, $3, -$2, int($4) }' \
				| csv-to-bin 3d,ui \
				> "$dirname/changedetection/$binname"
		done
	done
}

eval_underwood() {
	i=$1; shift
	angle_threshold=$(math-deg2rad $1); shift
	range_threshold=$1; shift

	path="${day}/${day}_drive_$(printf "%04d" $i)_sync/changedetection"

	numscans=$(ls "$path"/*.bin | wc -l)
	delta=5
	tmpdir=$(mktemp -d kitti.XXXXXXXXXX)
	numlines=$(wc -l < "${day}/${day}_drive_$(printf "%04d" $i)_sync/3dtk/slam6d.net")
	for linenum in $(seq 3 $numlines); do
		line=$(head -$linenum "${day}/${day}_drive_$(printf "%04d" $i)_sync/3dtk/slam6d.net" | tail -1)
		id1=$(echo "$line" | cut -d' ' -f 1)
		id2=$(echo "$line" | cut -d' ' -f 2)
#	for id1 in $(seq 0 $((numscans-1))); do
		scan1=$(printf "$path/scan%03d.bin" $id1)
		#echo "compute changes in: $scan1" >&2
		echo "compute changes for scenario $i between $id1 $id2" >&2
		dirname=$(dirname "$scan1")
		posename1=$(basename "$scan1" .bin).pose
		pose1=$(pose2csv < "$dirname/$posename1")
#		for id2 in $(seq $(($id1-$delta)) $(($id1+$delta))); do
			if [ "$id2" -lt 0 ] || [ "$id2" -gt "$(($numscans-1))" ]; then
				continue
			fi
			if [ "$id1" = "$id2" ]; then
				continue
			fi
			scan2=$(printf "$path/scan%03d.bin" $id2)
			posename2=$(basename "$scan2" .bin).pose
			pose2=$(pose2csv < "$dirname/$posename2")
			cat "$scan2" | points-to-polar --binary=3d,ui --fields=x,y,z > "$tmpdir/ref2rbe.bin"
			cat "$scan1" \
				| points-frame --binary=3d,ui --fields=x,y,z --from="$pose1" --to="$pose2" \
				| points-to-polar --binary=3d,ui --fields=x,y,z \
				| points-detect-change --binary=3d,ui --fields=r,b,e --angle-threshold="$angle_threshold" --range-threshold="$range_threshold" "$tmpdir/ref2rbe.bin" \
				| points-to-cartesian --binary=3d,ui --fields=r,b,e \
				| points-frame --binary=3d,ui --fields=x,y,z --from="$pose2" \
				> "$tmpdir/additions.bin"
			cat "$scan1" | points-frame --binary=3d,ui --fields=x,y,z --from="$pose1" > "$tmpdir/georef1.bin"
			addition_object_points_count=$(cat "$tmpdir/georef1.bin" | csv-eval --binary 3d,ui --fields x,y,z,label --output-if "label==1" | csv-from-bin 3d,ui | wc -l )
			addition_non_object_points_count=$(cat "$tmpdir/georef1.bin" | csv-eval --binary 3d,ui --fields x,y,z,label --output-if "label==0" | csv-from-bin 3d,ui | wc -l )
			addition_true_positives=$(cat "$tmpdir/additions.bin" | csv-eval --binary 3d,ui,3d,ui --fields x,y,z,label --output-if "label==1" | csv-from-bin 3d,ui,3d,ui | wc -l )
			addition_false_positives=$(cat "$tmpdir/additions.bin" | csv-eval --binary 3d,ui,3d,ui --fields x,y,z,label --output-if "label==0" | csv-from-bin 3d,ui,3d,ui | wc -l )
			echo "$addition_true_positives $addition_non_object_points_count $addition_false_positives $addition_object_points_count" \
			| awk '{tp=$1; fp=$3; tn=$2-fp; fn=$4-tp; p=(tp+fp==0?0:tp/(tp+fp));r=(tp+fn==0?0:tp/(tp+fn)); print tp, tn, fp, fn, p, r, (p+r==0?0:2*p*r/(p+r))}'
			rm "$tmpdir/additions.bin" "$tmpdir/georef1.bin" "$tmpdir/ref2rbe.bin";
#		done
	done
	rmdir "$tmpdir"
}

eval_3dtk() {
	i=$1; shift
	voxelsize=$1; shift
	path="${day}/${day}_drive_$(printf "%04d" $i)_sync/changedetection"
	tmpdir="$path/pplremover/masks/"
	mkdir -p "$tmpdir"

	peopleremover \
		-j16 \
		--maskdir="$tmpdir/" \
		--voxel-size=$voxelsize \
		--maxrange-method=normals \
		--normal-method=angle \
		--no-subvoxel-accuracy \
		"$@" \
		-f uosr \
		"$path" >/dev/null 2>&1
}

eval_all() {
	totalnumscans=0
	totalnumstatic=0
	totalnumdynamic=0
	for i in $ids; do
		printf "$i\t";
		dir=$(printf 2011_09_26/2011_09_26_drive_%04d_sync $i);
		numscans=$(head -1 $dir/3dtk/slam6d.net);
		numstatic=$(awk 'BEGIN{num=0}$3 == 0 {num++} END {print num}' $dir/changedetection/scan*.mask);
		numdynamic=$(awk 'BEGIN{num=0}$3 == 1 {num++} END {print num}' $dir/changedetection/scan*.mask);
		ratio=$(echo "scale=2;100*$numdynamic/($numstatic+$numdynamic)" | bc);
		printf "$numscans\t$ratio\n";
		totalnumscans=$((totalnumscans+numscans))
		totalnumstatic=$((totalnumstatic+numstatic))
		totalnumdynamic=$((totalnumdynamic+numdynamic))
	done > results0.txt
	ratio=$(echo "scale=2;100*$totalnumdynamic/($totalnumstatic+$totalnumdynamic)" | bc);
	printf "all\t$totalnumscans\t$ratio\n" >> results0.txt
	for i in $ids; do
		printf "$i ";
		eval_underwood $i 1.3 74 | collectstats
		echo
	done > results1.txt
	printf "all " >> results1.txt
	for i in $ids; do
		eval_underwood $i 1.3 74
	done | collectstats >> results1.txt
	for i in $ids; do
		echo "working on $i..." >&2
		eval_3dtk $i 39
	done
	python3 f1_3dtk.py $day $ids > results2.txt
	for i in $ids; do
		echo "working on $i..." >&2
		eval_3dtk $i 39 --min-cluster-size=210
	done
	python3 f1_3dtk.py $day $ids > results3.txt
	for i in $ids; do
		echo "working on $i..." >&2
		eval_3dtk $i 39 --reduce=10:0.2
	done
	python3 f1_3dtk.py $day $ids > results4.txt
	paste results0.txt results1.txt results2.txt results3.txt results4.txt \
		| awk -v OFS='\t' '{print $1, $2, $3, $5, $7, $11, $15, $12}'
}

if [ $# -lt 1 ]; then
	echo "usage: $0 action"
	echo
	echo "available actions:"
	echo
	echo "   download - downloads all data and code"
	echo "   compile  - compiles all code"
	echo "   convert  - converts all data"
	echo "   eval     - run change detection"
	echo "   all      - all of the above"
	exit 1
fi

script=$0
mode=$1
shift
case $mode in
	eval_3dtk)
		eval_3dtk "$@";;
	eval_underwood)
		eval_underwood "$@";;
	convert)
		convert;;
	download)
		download;;
	compile)
		compile;;
	clean)
		clean;;
	eval)
		eval_all "$@";;
	all)
		download
		compile
		convert
		eval_all
		;;
	*)
		echo "invalid mode: $mode" >&2;;
esac
